// **************************************************************************
// * File:   Sockets.cpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Code to implement socket functionality							*
// * Notes:  The target project will need to have 'wsock32.dll' in the link	*
// *         libraries setting under Microsoft.								*
// **************************************************************************

#include "Ertsys.hpp"

#include <cstdlib>

#if defined(_MSC_VER)

 #define WIN32_LEAN_AND_MEAN
 #if defined(_dUnicode)
  #define UNICODE	(1)		// used by some of the MS header files
 #endif

 #include <winsock2.h>
 #define LENPARAM(name) (int FAR*)&name

#else

 #include <sys/types.h>
 #include <sys/socket.h>
 #include <netdb.h>
 #include <unistd.h>		// for 'close'
 #include <arpa/inet.h>		// for 'inet_addr' and 'inet_ntoa' (could have used '<netinet/in.h>')

 extern int h_errno;
 extern int errno;

 #define SOCKET_ERROR (-1)
 #define INVALID_SOCKET (-1)
 #define closesocket close
 #define LENPARAM(name) (socklen_t*)&name

#endif

#include "Sockets.hpp"
#include "CharHelpers.hpp"

// This constant defines the backlog parameter for server sockets. This is the maximum
// length the queue of pending connections may grow to. If a connection request
// arrives with the queue full the client may receive an error with an indication
// of ECONNREFUSED or, if the underlying protocol supports retransmission, the request
// may be ignored so that retries may succeed
const int CONN_BACKLOG = 5;


// Environment methods to open sockets. Moved these to this file to try to reduce executable size in projects
// which don't use sockets. Nigel, 21-3-02.

void Environment :: open (const _eSeq < _eChar > hostname, const _eInt port, const SocketMode::_eEnum sktMode, _eUnion & s)
{
    _mSchema (open);
    _mDontShare (this);
	_eSocket *skt = new _eSocket (hostname, port, sktMode);
	if (skt->ready())
	{
		s = static_cast < const _eAny * > (skt);
	}
	else
	{
    	s = _mWrapStorableEnum (skt->getLastError(), SocketError);
	}
}

void Environment :: open (const _eSeq < _eByte > ipAddress, const _eInt port, const SocketMode::_eEnum sktMode, _eUnion & s)
{
    _mSchema (open);
    _mDontShare (this);
	_eSocket *skt = new _eSocket (ipAddress, port, sktMode);
	if (skt->ready())
	{
		s = static_cast < const _eAny * > (skt);
	}
	else
	{
    	s = _mWrapStorableEnum (skt->getLastError(), SocketError);
	}
}

#if defined(_MSC_VER)
	// Class exists purely so that when the program exits, its destructor is called and we clean up the sockets (Nigel 21-03-02)
	class SocketCleaner
	{	public:
			bool initialized;

			SocketCleaner() : initialized(false) {}

			~SocketCleaner() { if(initialized) WSACleanup(); }
	};

	// This will be destroyed on program exit, thus cleaning up any sockets
	SocketCleaner socketCleaner;
#endif

// Private method to initialize socket services on the host machine. Only relevant on the MSW platform. Sets
// the 'seError' variable on failure, but doesn't clear it on success
bool _eSocket::initializeWinsock()
{
  #if defined(_MSC_VER)
	if (!socketCleaner.initialized)
	{
		WSADATA wsaData;
		if (WSAStartup(0x202, &wsaData) == SOCKET_ERROR)
		{
			seError = SocketError::initError;
			WSACleanup();
			return false;	// failed!
		}
		socketCleaner.initialized = true;
	}
  #endif
	return true;
}

// Private method to actually open and connect-to/listen-on a TCP socket - called by the CTORs of this
// class only. Sets the 'seError' variable on failure, but doesn't clear it on success
bool _eSocket::OpenTcpSocket(struct sockaddr_in &socketData, _eInt portNumber, bool bServerMode)
{
	// Open a TCP socket
	conn_socket = socket(AF_INET, SOCK_STREAM, 0);
	if (conn_socket == INVALID_SOCKET)
	{
		seError = SocketError::invalidSocket;			// to be more like Java, this probably should be IOError
		return false;
	}
	if (bServerMode)
	{	// In server mode, ie. we want to attempt to set up the socket for 'listening' (i.e. we want to
		// be able to accept connections on the port) ...
		if(bind(conn_socket, (struct sockaddr*)&socketData, sizeof(socketData)) == SOCKET_ERROR)
		{
			seError = SocketError::ioError;
			//int detailedError = WSAGetLastError();	// ... or use 'errno' on Linux
			return false;
		}
		if (listen(conn_socket, CONN_BACKLOG) == SOCKET_ERROR)
		{
			seError = SocketError::ioError;
			//int detailedError = WSAGetLastError();	// ... or use 'errno' on Linux
			return false;
		}
	}
	else
	{	// In client mode, ie. we want to attempt to connect to the specified socket that is accepting
		// connection (i.e. a server must be running on the specified port on the specified host) ...
		if (connect(conn_socket, (struct sockaddr*)&socketData, sizeof(socketData)) == SOCKET_ERROR)
		{
			seError = SocketError::ioError;
			//int detailedError = WSAGetLastError();	// ... or use 'errno' on Linux
			return false;
		}
	}
	return true;
}

// Constructor to generate a socket connected to the specified port number at the specified IP address (as
// a sequence of 4 nats). Sets the 'seError' variable on failure, and clears it on success
_eSocket::_eSocket(_eSeq< _eByte > ipAddress, _eInt portNumber, const SocketMode::_eEnum sktMode)
	: smMode(sktMode)
{
	_mBeginPre _mCheckPre(ipAddress._oHash() == 4, "95,53");
	// generate a string in the dotted decimal format (eg. 192.202.27.1)
	_rstring dottedAddress = _ltoString(ipAddress[0]);
	for (int i=1; i<4; i++)
	{
		dottedAddress = dottedAddress._oPlusPlus(_mString("."))._oPlusPlus(_ltoString(ipAddress[i]));
	}
  #if defined(_MSC_VER)
	WSASetLastError(0);			// clearout any irrelevant error code
  #endif
	if (initializeWinsock())
	{
		// generate a host-entry pointer ready for opening the socket
#if _dUnicode
		char* p;
		wcs2ncs(p, _eCstring(dottedAddress).str());
		unsigned long addr = inet_addr(p);
#else
		unsigned long addr = inet_addr(_eCstring(dottedAddress).str());
#endif
		if (addr == INADDR_NONE)
		{
			seError = SocketError::invalidAddress;
		}
		else
		{	// Setup the socket data structure with the IP address, the port number and the socket family info
			struct sockaddr_in socketData;
			memset(&socketData, 0, sizeof(socketData));
			memcpy(&(socketData.sin_addr), &addr, sizeof(addr));
			socketData.sin_family = AF_INET;
			socketData.sin_port = htons(static_cast<unsigned short>(portNumber));		// 'htons' ensures port number is in network byte-order
			// OK, attempt to open and connect a socket ...
			if (OpenTcpSocket(socketData, portNumber, sktMode == SocketMode::server))	// will set 'seError' as appropriate on failure
			{
				seError = SocketError::success;
				bSocketReady = true;
				return;
			}
		}
	}
	bSocketReady = false;
}

// Constructor to generate a socket connected to the specified port number at the specified host (the
// 'MY COMPUTER' name). Sets the 'seError' variable on failure, and clears it on success
_eSocket::_eSocket(const _rstring &hostName, _eInt portNumber, const SocketMode::_eEnum sktMode)
	: smMode(sktMode)
{
	initializeWinsock();
  #if defined(_MSC_VER)
	WSASetLastError(0);			// clearout any irrelevant error code
  #endif
	bSocketReady = false;
	if (initializeWinsock())
	{
#if defined(_dUnicode)
		char* pHostname;
		wcs2ncs(pHostname, _eCstring(hostName).str());
		struct hostent *hp = gethostbyname(pHostname);
#else
		struct hostent *hp = gethostbyname(_eCstring(hostName).str());
#endif
		if (hp == NULL )
		{
	  	 #if defined(_MSC_VER)
			if (WSAGetLastError() == WSAHOST_NOT_FOUND)
	  	 #else
			if (h_errno == HOST_NOT_FOUND)
	  	 #endif
	  		{
				seError = SocketError::unknownHost;
			}
			else
			{
				seError = SocketError::generalError;	// to be more like Java, this probably should be IOError
			}
		}
		else
		{	// Resolved DNS name OK, so copy the resolved information into the sockaddr_in structure
			struct sockaddr_in socketData;
			memset(&socketData, 0, sizeof(socketData));
			memcpy(&(socketData.sin_addr), hp->h_addr, hp->h_length);
			socketData.sin_family = hp->h_addrtype;
			socketData.sin_port = htons(static_cast<unsigned short>(portNumber));		// ensure port number is in network byte-order
			// Now attempt to open and connect a socket ...
			if (OpenTcpSocket(socketData, portNumber, sktMode == SocketMode::server))	// will set 'seError' as appropriate on failure
			{
				seError = SocketError::success;
				bSocketReady = true;
				return;
			}
		}
	}
	bSocketReady = false;
}

_eSocket::~_eSocket()
{
	// IMPORTANT NOTE:
	// This destructor used to call 'closesocket(conn_socket)' but this was wrong since this class
	// holds what amounts to a reference to the actual socket (the local 'conn_socket' variable is
	// an int that represents a handle to the system-wide socket), and thus we mustn't make it
	// invalid locally by closing it here because of the way the handle logic works (if a handle to
	// a socket gets copied and then later goes out of scope, the socket DTOR will get called for
	// that instance, and as a result, any subsequent operations on the socket would fail)
}

// Wait for a connection to this server socket (ie. block), and accept the first one that comes along. If all is well,
// set the result value to true, to indictate that a user may now call 'read(..)' on the socket to obtain the data
// waiting. If there is some problem, return false and store an appropriate error (if we are not a server socket, we
// just return straight away after setting the appropriate error). The connection established by this call can be
// terminated by a call to 'closeConnection(..)'
void _eSocket::awaitConnection(bool &res)
{
	if (smMode != SocketMode::server)
	{
		seError = SocketError::wrongType;
		return;
	}
	// If the previously used socket is still open, close it before we accept another connection
	if (msgsock != -1)
	{
		closesocket(msgsock);
	}
	// Wait for a connection ...
	sockaddr_in from;
	int from_len = sizeof(from);
	msgsock = accept(conn_socket,(struct sockaddr*)&from, LENPARAM(from_len));
	if (msgsock == INVALID_SOCKET)
	{
		seError = SocketError::ioError;
		//int detailedError = WSAGetLastError();	// ... or use 'errno' on Linux
		res = false;
		return;
	}
	// OK, sucessfully accepted a connection, so note the remote IP address and return success
	const char* p = inet_ntoa(from.sin_addr);
#if defined(_dUnicode)
	_eNativeChar* buf;
	ncs2wcs(buf, p);
	remoteAddress = _econvertIPstring(_lString(buf));
#else
	remoteAddress = _econvertIPstring(_lString(p));
#endif
	remotePort = htons(from.sin_port);	// don't think we really need this
	seError = SocketError::success;
	res = true;
}

// Schema to allow user to close the connection socket established by the last call to 'awaitConnection(..)'. If we
// are not a server socket, return failed)
void _eSocket::closeConnection(bool &res)
{
	if (smMode != SocketMode::server)
	{	// We're not a server socket!
		seError = SocketError::wrongType;
		res = false;
	}
	else if (!bSocketReady || msgsock == -1)
	{	// A connection socket isn't currently open
		seError = SocketError::connectionNotOpen;
		res = false;
	}
	else
	{	// We are a server socket and there _is_ a connection open, so attempt to close it
		if (closesocket(msgsock))
		{	// Failed to close the socket connection
			seError = SocketError::ioError;		// very generic error, like Java
			res = false;
		}
		else
		{	// Closed the connection OK
			bSocketReady = false;
			seError = SocketError::success;
			res = true;
		}
	}
}

// Write to the socket. Sets the 'seError' variable on failure, and clears it on success
void _eSocket::write(const _rstring &wdata, bool &res)
{
	_eCstring temp(wdata);
	SOCKET theSocket = (smMode == SocketMode::client) ? conn_socket : msgsock;
#if defined(_dUnicode)
	char* ptemp;
	wcs2ncs(ptemp, temp.str());
  #if defined(_WIN64)
	//??? we should loop if necessary here
	if (send(theSocket, ptemp, static_cast<int>(temp.strlen()), 0) != SOCKET_ERROR)
  #else
	if (send(theSocket, ptemp, temp.strlen(), 0) != SOCKET_ERROR)
  #endif
#else
  #if defined(_WIN64)
	//??? we should loop if necessary here
	if (send(theSocket, temp.str(), static_cast<int>(temp.strlen()), 0) != SOCKET_ERROR)
  #else
	if (send(theSocket, temp.str(), temp.strlen(), 0) != SOCKET_ERROR)
  #endif
#endif
	{
		seError = SocketError::success;
	}
	else
	{
		seError = SocketError::ioError;		// very generic error, like Java
	}
}

// Read data from the socket. Sets the 'seError' variable on failure, and clears it on success
void _eSocket::read(_eSeq <_eByte > &rdata, bool &res)
{
	const int DefaultBufSize = 512;
	char *temp = (char *)_eMem::alloc(DefaultBufSize);		// buffer with a sensible default size
	SOCKET theSocket = smMode == SocketMode::client ? conn_socket : msgsock;
	int recvRes;
	if ((recvRes = recv(theSocket, temp, DefaultBufSize, 0)) == SOCKET_ERROR)
	{
		int err;
	  #if defined(_MSC_VER)
		err = WSAGetLastError();
		if (err == WSAEMSGSIZE)
	  #else
		err = h_errno;
		if (false)//err == HOST_NOT_FOUND)	//??? Not sure how to detect this situation under Linux...
	  #endif
		{
			// The data waiting is too big to fit in the temp buffer we've allocated, so we need to find out just
			// how big it is so we can re-allocate the buffer
			//??? This is horribly inefficient, so it would best to make sure that the default buffer is large
			//    enough that any message will (nearly) always fit into it first time
//			printf("[Message too big, so re-allocating buffer and re-reading...]\n");	// DEBUG!!!
			int msgLen = recv(theSocket, temp, DefaultBufSize, MSG_PEEK);
			temp = (char *)_eMem::alloc(msgLen);		// re-allocate the read buffer - don't need to free the old since freelists will do it for us)
			if ((recvRes = recv(theSocket, temp, msgLen, 0)) == SOCKET_ERROR)
			{
				res = false;
				seError = SocketError::ioError;		// very generic error, like Java
				return;
			}
			else if (recvRes == 0)	// was connection 'gracefully' closed?
			{
				res = false;
				seError = SocketError::connectionNotOpen;
				return;
			}
		}
		else
		{
			res = false;
		  #if defined(_MSC_VER)
			seError = (err == WSAENOTCONN || err == WSAECONNABORTED || err == WSAETIMEDOUT || err == WSAECONNRESET) ?
					  SocketError::connectionNotOpen : SocketError::ioError;			// very generic error, like Java
		  #else
			// Don't know how to find out more specifically what the error was under Linux ...
			seError = SocketError::ioError;			// very generic error, like Java
		  #endif
			return;
		}
	}
	else if(recvRes == 0)	// was connection 'gracefully' closed?
	{
		res = false;
		seError = SocketError::connectionNotOpen;
		return;
	}
	res = true;
	seError = SocketError::success;
	// Finally, populate the Perfect seqence
	for (int i=0; i<recvRes; i++)
	{
		rdata = rdata.append(static_cast<_eByte>((unsigned char)temp[i]));
	}
}

// Read data if any is available but don't block if not, just return an empty string instead.
// Sets the 'seError' variable on failure, and clears it on success.
//??? This is not very efficient, since it causes any data waiting to be read from the recv
//    buffer twice; necessary since there's no way I can find to just ask for the size of any
//    waiting data without doing a MSG_PEEK (the peek actually causes the data to be copied
//    into the buffer passed, but we need to read it without peeking so as to actually remove
//    it from the input buffer - again, there's no other way I can find to do this)
//??? WARNING: Under winsock 1.1, a 'bug' exists that could cause this function to always
//    return no-data. See MSDN article "MSG_PEEK Always Returns Wrong Buffer Size" for details.
void _eSocket::read_noblock(_eSeq <_eByte > &rdata, bool &res)
{
	const int DefaultBufSize = 512;
	char *temp = (char *)_eMem::alloc(DefaultBufSize);		// buffer with a sensible default size
	SOCKET theSocket = smMode == SocketMode::client ? conn_socket : msgsock;
	int dataLen = recv(theSocket, temp, DefaultBufSize, MSG_PEEK);
	if (dataLen > 0)
	{	// There is data waiting, so read it expanding the buffer first if necessary
		if (dataLen > DefaultBufSize)
		{	// Need to allocate a larger buffer in order to recieve all the data - don't need to free the old since
			// freelists will do it for us)
			temp = (char *)_eMem::alloc(dataLen);
		}
		{
			const int recvRes = recv(theSocket, temp, dataLen, 0);
			if (recvRes == SOCKET_ERROR)
			{
				res = false;
				seError = SocketError::ioError;		// very generic error, like Java
				return;
			}
			else if (recvRes == 0)	// was connection 'gracefully' closed?
			{
				res = false;
				seError = SocketError::connectionNotOpen;
				return;
			}
		}
		// OK, populate the Perfect seqence
		for (int i = 0; i < dataLen; i++)
		{
			rdata = rdata.append(static_cast<_eByte>((unsigned char)temp[i]));
		}
	}
	else
	{	// No data waiting, so just return an empty sequence ...
		rdata = _eSeq <_eByte >();
	}
	seError = SocketError::success;
	res = true;
}

// Read the specified number of 8-bit bytes from the socket specified. Blocks until all bytes read or
// an error is encountered. Return the sequence of bytes read, in the order read, or an appropriate socket error
void _eSocket::read(const _eInt numBytes, _eUnion &res)
{
	_mSchema(scan);
	_mBeginPre _mCheckPre(numBytes > 0, "0,0");
	_eSeq <_eByte > resTemp;
	char *temp = new char[numBytes];
	SOCKET theSocket = (smMode == SocketMode::client) ? conn_socket : msgsock;
	int rres;
	_eInt bytesRead = 0;

	// Read the specified number of bytes
	while (bytesRead < numBytes)
	{
#if defined(_WIN64)
		const _eInt bytesLeft = numBytes - bytesRead;
		const int bytesToRead = (bytesLeft > INT_MAX) ? INT_MAX : static_cast<int>(bytesLeft);
#else
		const int bytesToRead = numBytes - bytesRead;
#endif
		if ((rres = recv(theSocket, temp + bytesRead, bytesToRead, 0)) == SOCKET_ERROR)
		{	// Socket read failed
			delete[] temp;
    		res = _mWrapStorableEnum(SocketError::ioError, SocketError);	// very generic error, like Java.
			return;
		}
		else if (rres == 0)
		{	// Socket was closed 'gracefully'
			delete[] temp;
    		res = _mWrapStorableEnum(SocketError::connectionNotOpen, SocketError);
			return;
		}
		else
		{	// Sucessfully read 'rres' bytes from the socket
			bytesRead += rres;
		}
	}

	// OK, populate the Perfect seqence
	for (_eInt i = 0; i < numBytes; i++)
	{
		resTemp = resTemp.append(static_cast<_eByte>(temp[i]));
	}

	delete[] temp;
	res = _mWrapStorable(resTemp, _eSeq < _eByte >);						// success!!
}

// Write the specified 8-bit byte sequence to the socket specified. Returns an appropriate
// socket error if fails, or None@SocketError if suceeds
void _eSocket :: write (const _eSeq < _eByte > data, SocketError :: _eEnum & res)
{
    _mSchema (write);
	_mBeginPre _mCheckPre(data._oHash() > 0, "0,0");
	SOCKET theSocket = (smMode == SocketMode::client) ? conn_socket : msgsock;

	const _eInt length = data._oHash();

	// Build a byte buffer from the Perfect sequence data ...
	char *tmpBuf = new char[length];

	for (_eInt i = 0; i < length; i++)
	{
		tmpBuf[i] = static_cast<char>(data[i]);
	}

	// Send the data off to the socket
#if defined(_WIN64)
	//??? we should loop here if needed...
	const int err = send(theSocket, tmpBuf, static_cast<int>(length), 0);
#else
	const int err = send(theSocket, tmpBuf, length, 0);
#endif
	delete[] tmpBuf;

	if (err != SOCKET_ERROR)
	{
		res = SocketError::success;
	}
	else
	{
	  #if defined(_MSC_VER)
		res = (err == WSAENOTCONN || err == WSAECONNABORTED || err == WSAETIMEDOUT || err == WSAECONNRESET)
			 ? SocketError::connectionNotOpen
			 : SocketError::ioError;			// very generic error, like Java
	  #else
		// Don't know how to find out more specifically what the error was under Linux ...
		res = SocketError::ioError;				// very generic error, like Java
	  #endif
	}
}

// Read a single byte from the socket
void _eSocket :: read (_eUnion & res)
{
    _mSchema (read);
	// just call the sequence version for now - re-implement for efficiency later ...
    _eUnion tmp;
    read (1, tmp);
    if (_mWithinNonHandle(tmp.Ptr(), SocketError::_eEnum))
    {
        res = _mWrapStorableEnum(_mIsNonHandle(tmp.Ptr(), SocketError::_eEnum), SocketError);
    }
    else
    {
        res = _mWrapBasic(_mIsNonHandle(tmp.Ptr(), _eSeq<_eInt>).head(), _eInt);
    }
}

// Write a single byte to the socket
void _eSocket :: write (const _eByte data, SocketError :: _eEnum & res)
{
    _mSchema (write);
    write(_eSeq<_eByte>(data), res);	// just call the sequence version for now - re-implement for efficiency later.
}

// Function to obtain the last network error recorded (or 'None' if last operation suceeded) ...
SocketError::_eEnum _eSocket::getLastError() const
{
	return seError;
}

// Schema to allow user to close the socket when he has finished with it. Note that if the socket wasn't open
// when this function was called, we return sucess (boolean true) anyway. Sets the 'seError' variable on failure,
// and clears it on success
void _eSocket::closeSocket(bool &res)
{
	if (bSocketReady)
	{
		// If we are a server and the previously used socket is still open, close it before we close the manager socket ...
		if (smMode == SocketMode::server && msgsock != -1)
		{
			closesocket(msgsock);
			//??? Maybe should process possible errors here...
		}
		if (closesocket(conn_socket))
		{	// Failed to close the socket!
			res = false;
			seError = SocketError::ioError;		// very generic error, like Java
			return;
		}
		bSocketReady = false;
	}
	seError = SocketError::success;
	res = true;
}

_eHndl < _eInstblTypeInfo > _eSocket :: _aMyTypeInfo ()
{
    static _eHndl<_eInstblTypeInfo> ti;
    ti = _eHndl<_eInstblTypeInfo>(new _eInstblTypeInfo(_estcBasicTypeCode :: BUILTIN_TYPEIDX_SOCKET, false));
    return ti;
}

// End.
