// **************************************************************************
// * File:   MiscRts.hpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Miscellaneous runtime functions.								*
// **************************************************************************

#if !defined(_h_MiscRts)
#define _h_MiscRts 1

// The general rule here is that we try to avoid #including files that pollute the global namespace,
// therefore we only inline functions that don't require extra #includes.

//----------------------------------------------------------------------------
// Macros for complex global constant declarations.
// Note that code to assign a value to "_Name" goes between these two macros
//----------------------------------------------------------------------------

#if _dCallStack
	#define _mBeginConstant(_Name, _Type)				\
		{	static _eMethod _tSelf(_mCstr("con: ") _mCstr(#_Name), _mCurrentFile, __LINE__); \
			auto _eCallStack _tCall(&_tSelf);			\
			static bool initialised = false;			\
			static _Type _Name;							\
			if (!initialised)							\
			{
#else
	#define _mBeginConstant(_Name, _Type)				\
		{	static bool initialised = false;			\
			static _Type _Name;							\
			if (!initialised)							\
			{
#endif

#define _mEndConstant(_Name)						\
            initialised = true;                     \
		}                                           \
		return _Name;								\
	}

// Macro to implement a specialization of the global template function used by the automatically generated
// 'myTypeInfo()' functions for builtin types (basics, enums etc.) (see 'LoadableSupport_1.hpp') ...
#define _mBuiltinTypeInform(type, builtinIndex)	\
	template<> struct _atypeInform<type >	\
	{	\
		static _eHndl<_eInstblTypeInfo> get()	\
		{	\
			return _eHndl<_eInstblTypeInfo>(new _eInstblTypeInfo(builtinIndex, false));	\
		}	\
	};

// Macro to implement a specialization of the global template function used by the automatically
// generated 'myTypeInfo()' functions for user enumerated types ...
#define _mUserTypeInform(type, nodeAddressCode, nodeIndex)	\
	template<> struct _atypeInform<type >	\
	{	\
    	static _eHndl<_eInstblTypeInfo> get()	\
    	{	\
        	static _eHndl<_eInstblTypeInfo> ti = _eHndl<_eInstblTypeInfo>(new _eInstblTypeInfo(	\
            	_eModuleDescriptorAddress(nodeAddressCode), nodeIndex));	\
        	return ti;	\
    	}	\
	};

//-------------------------------------------------------------------
// Member operators for rank and complex value classes
//-------------------------------------------------------------------

namespace _eRank{_mEnumMembers}
_mEnumGlobalSupport(_eRank)

namespace _aExceptionId{_mEnumMembers}
_mEnumGlobalSupport(_aExceptionId)

namespace _eComplexValue{_mEnumMembers}
_mEnumGlobalSupport(_eComplexValue)

//---------------------------------------------
// Member functions of class "byte"
//---------------------------------------------

// These definitions rely on the C++ type _eByte being an unsigned type! (see Types.hpp)

// Successor operator on byte
inline _eByte _oSucc(const _eByte x)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (x != 255, "0,0");
    #endif
	return static_cast<_eByte>(x + 1);
}

// Predecessor operator on byte
inline _eByte _oPred(const _eByte x)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (x != 0, "0,0");
    #endif
	return static_cast<_eByte>(x - 1);
}

// Unary + operator for byte
inline _eInt _oPlus(const _eByte x)
{
	return static_cast<_eInt>(x);
}

// ".." operator for byte
inline _eSeq<_eByte> _oRange(const _eByte x, const _eByte y)
{
	_eSeq<_eByte> result = _eSeq<_eByte>();
	for (_eByte i = x; i <= y; i++)
		result = result._lAppend(i);
	return result;
}

inline _eByte _lbyteAnd(_eByte a, _eByte b)
{
	return static_cast<_eByte>(a & b);
}

inline _eByte _lbyteOr(_eByte a, _eByte b)
{
	return static_cast<_eByte>(a | b);
}

inline _eByte _lbyteXor(_eByte a, _eByte b)
{
	return static_cast<_eByte>(a ^ b);
}

inline _eByte _lbyteCompl(_eByte a)
{
	return static_cast<_eByte>(~a);
}

inline _eByte _lbyteShl(_eByte a, _eInt b)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (0 <= b && b < 8, "0,0");
    #endif
	return static_cast<_eByte>(a << b);
}

inline _eByte _lbyteShr(_eByte a, _eInt b)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (0 <= b && b < 8, "0,0");
    #endif
	return static_cast<_eByte>(a >> b);
}

// Byte constructor taking a sequence of bits, most significant bit first
//	build name "_lBitsToByte" {!theBits: seq of bool}
//		pre #theBits = 8;
extern _eByte _lBitsToByte(const _eSeq<_eBool>);

// Byte constructor taking an integer argument
// We name the constructor so that we can implement a precondition check
//	build name "_lNatToByte" {arg: nat}
//		pre arg < 256
//		^= 	byte
//			{	seq of bool
//				{	(arg)       >= 128,
//					(arg % 128) >= 64,
//					(arg % 64)  >= 32,
//					(arg % 32)  >= 16,
//					(arg % 16)  >= 8,
//					(arg % 8)   >= 4,
//					(arg % 4)   >= 2,
//					(arg % 2)    = 1
//				}
//			}
inline _eByte _lNatToByte(_eInt arg)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (0 <= arg && arg < 256, "0,0");
    #endif
	return static_cast<_eByte>(arg);
}

//---------------------------------------------
// Member functions of class "char"
//---------------------------------------------

// Successor operator on char
inline _eChar _oSucc(const _eChar x)
{
	return static_cast<_eChar>(static_cast<_eUnsignedChar>(x) + 1);
}

// Predecessor operator on char
inline _eChar _oPred(const _eChar x)
{
	return static_cast<_eChar>(static_cast<_eUnsignedChar>(x) - 1);
}

// Unary + operator for char
inline _eInt _oPlus(const _eChar x)
{
	return static_cast<_eInt>(static_cast<_eUnsignedChar>(x));
}

// ".." operator for char
inline _eSeq<_eChar> _oRange(const _eChar x, const _eChar y)
{
	_eSeq<_eChar> result = _eSeq<_eChar>();
	for (_eUnsignedChar i = x; i <= static_cast<_eUnsignedChar>(y); i++)
	{
		result = result._lAppend(static_cast<_eChar>(i));
	}
	return result;
}

//	function isLetter name "_lisLetter": bool ^= ?;
extern _eBool _lisLetter(_eChar c);

//	function isLowerCase name "_lisLowerCase": bool ^= ?;
extern _eBool _lisLowerCase(_eChar c);

//	function isUpperCase name "_lisUpperCase": bool ^= ?;
extern _eBool _lisUpperCase(_eChar c);

//	function isDigit name "_lisDigit": bool ^= ?;
extern _eBool _lisDigit(_eChar c);

//	function isPrintable name "_lisPrintable": bool ^= ?;
extern _eBool _lisPrintable(_eChar c);

//	function digit name "_ldigit": int pre isDigit ^= ? assert 0 <= result < 10;
inline _eInt _ldigit(_eChar c) { return static_cast<_eInt>(c - _mChar('0')); }

//	function toLowerCase name "_ltoLowerCase": char ^= ? assert ~self.isUpperCase ==> result = self;
extern _eChar _ltoLowerCase(_eChar c);

//	function toUpperCase name "_ltoUpperCase": char ^= ? assert ~self.isLowerCase ==> result = self;
extern _eChar _ltoUpperCase(_eChar c);

//---------------------------------------------
// Member functions of class "int"
//---------------------------------------------

// Successor operator on integer
inline _eInt _oSucc(const _eInt x)
{
	return x + 1;
}

// Predecessor operator on integer
inline _eInt _oPred(const _eInt x)
{
	return x - 1;
}

// ".." operator for integer
inline _eSeq<_eInt> _oRange(const _eInt x, const _eInt y)
{
	_eSeq<_eInt> result = _eSeq<_eInt>();
	for (_eInt i = x; i <= y; i++)
		result = result._lAppend(i);
	return result;
}

// Integer exponentiation function
extern _eInt _oExp(_eInt aa, _eInt bb);

// Integer division function
inline _eInt _oDiv(const _eInt aa, const _eInt bb)
{
	return aa >= 0 ? aa / bb : -((-aa - 1)/bb + 1);
}

// Integer modulus function
inline _eInt _oMod(const _eInt aa, const _eInt bb)
{
	return aa >= 0 ? aa % bb : bb - 1 - ((-aa - 1) % bb);
}

inline void _osSlash(_eInt& aa, const _eInt bb)
{
	aa = _oDiv(aa, bb);
}

inline void _osPercent(_eInt& aa, const _eInt bb)
{
	aa = _oMod(aa, bb);
}

inline _eInt _labs(const _eInt aa)
{
	return (aa < 0) ? -aa : aa;
}

// Integer approx log function
extern _eInt _lintln(_eInt);

// Bit operations
inline _eInt _lintAnd(_eInt a, _eInt b)
{
	return a & b;
}

inline _eInt _lintOr(_eInt a, _eInt b)
{
	return a | b;
}

inline _eInt _lintXor(_eInt a, _eInt b)
{
	return a ^ b;
}

inline _eInt _lintCompl(_eInt a)
{
	return ~a;
}

inline _eInt _lintShl(_eInt a, _eInt b)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (0 <= b, "0,0");
    #endif
	return a << b;
}

inline _eInt _lintShr(_eInt a, _eInt b)
{
    #if !defined(NDEBUG)
    _mBeginPre _mCheckPre (0 <= b, "0,0");
    #endif
	return a >> b;
}

//---------------------------------------------
// Member functions of class "real"
//---------------------------------------------

extern _eInt _lroundup(const _eReal aa);
extern _eInt _lrounddn(const _eReal aa);

inline _eInt _lroundin(const _eReal aa)
{
	return static_cast<_eInt>(aa);
}

inline _eInt _lroundout(const _eReal aa)
{
	return aa >= 0 ? _lroundup(aa) : _lrounddn(aa);
}

inline _eBool _lisNaN(const _eReal aa)
{
	return aa != aa;
}

extern _eBool _lisInfinite(const _eReal aa);

extern _eReal _oExp(_eReal aa, _eReal bb);
extern _eReal _oExp(_eInt aa, _eReal bb);
extern _eReal _oExp(_eReal aa, _eInt bb);

inline _eReal _oDiv(const _eReal aa, const _eReal bb)
{
	return aa / bb;
}

inline _eReal _oDiv(const _eInt aa, const _eReal bb)
{
	return static_cast<_eReal>(aa) / bb;
}

inline _eReal _oDiv(const _eReal aa, const _eInt bb)
{
	return aa / static_cast<_eReal>(bb);
}

//---------------------------------------
// Temporary exception throwing mechanism
//---------------------------------------

extern _eBool _aThrow (const _eEselException *);

// Class used by '_aThrow(..)' to wrap the exception into something we can throw in C++
class _aEselExceptionWrapper
{
public:
    // CONCRETE DIRECT class
    _mNew (_aEselExceptionWrapper)
private:
    _eFrom < _eEselException > exception;
public:
    virtual _eSeq < _eChar > msg () const;
public:
    virtual _aExceptionId :: _eEnum id () const;
public:
    explicit _aEselExceptionWrapper (const _eEselException *);
public:
    _eBool operator == (const _aEselExceptionWrapper &) const;
    _aEselExceptionWrapper () {}
    virtual ~_aEselExceptionWrapper() {}
};

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
