#!/bin/bash
# Shell script post-build step command file for building a Java application from Perfect Developer version 3

# Check that at least two command-line parameters have been supplied
if [ -z $2 ]; then
echo "Post-build file: not enough parameters provided!"
echo "Usage:    postbuildjava.bat  main-class-name library-name  additional-java-files"
echo "Example:  postbuildjava.bat  Entry PerfectRuntime.jar  Entry.java"
exit 1
fi

MAINCLASS=$1

# The Project Manager (version 3 or later) sets up PD_JAVA_LIB_PATH before executing the post-build step. Check for this.
if [ -z "${PD_JAVA_LIB_PATH}" ]; then
echo "Post-build file: variable PD_JAVA_LIB_PATH has not been set up! Use this file only with Perfect Developer version 3 or later."
exit 1
fi

# The Project Manager also sets up PD_JDK_BIN_PATH provided the user has configured it. Check for this.
if [ -z "${PD_JDK_BIN_PATH}" ]; then
echo "Post-build file: Path to Java tools not configured! Please configure the path to the Java tools in the Project Manager 'Options' page."
exit 1
fi

# Check that the Java tools path is set up by looking for the Java compiler
if [ ! -f ${PD_JDK_BIN_PATH}javac ]; then
echo "Post-build file: Java compiler not found on path \"${PD_JDK_BIN_PATH}\"! Please configure the path to the Java tools in the Project Manager 'Options' page"
exit 1
fi

# Build the complete library filename from the path variable and the supplied filename, then check that it exists
PERFECTLIB=${PD_JAVA_LIB_PATH}$2
if [ ! -f ${PERFECTLIB} ]; then
echo "Post-build file: incorrect library file and/or path name (can't find file \"${PERFECTLIB}\")"
exit 1
fi

# Check that a package name has been set up
if [ -z "${PD_PACKAGE_NAME}" ]; then
echo "Post-build file: You must set a package name in the project to use this file!"
exit 1
fi

# Check that there are some generated java source files where we expect them
shopt -s nullglob
JAVAFILES=./src/${PD_PACKAGE_NAME}/*.java
if [ -z "${JAVAFILES}" ]; then
echo "Post-build file: PD_PACKAGE_NAME setting is incorrect (can't find any .java files in ./src/${PD_PACKAGE_NAME})"
exit 1
fi

# Checks complete, now perform the post-build

# Create the java classes directory if it doesn't exist
if [ ! -d classes/$PKGNAME ]; then
  mkdir classes/$PKGNAME
fi

# Create the output directory if it doesn't exist
if [ ! -d output ]; then
  mkdir output
fi

echo "Compiling Java files for package $PKGNAME"
"${PD_JDK_BIN_PATH}javac" -d ./classes -classpath "./classes:./src:${PERFECTLIB}" ./src/${PD_PACKAGE_NAME}/*.java $3 $4 $5 $6
if [ $? -ne 0 ]; then
  exit 1
fi

# Copy the library archive to the output directory, changing its name to the name of our package
echo "Copying library"
cp "${PERFECTLIB}" output/${PD_PACKAGE_NAME}.jar
if [ $? -ne 0 ]; then
  echo "copy failed!"
  exit 1
fi

# Add our own class files to the archive and delete its manifest, if any
echo "Creating JAR file"
"${PD_JDK_BIN_PATH}jar" ufM0 output/${PD_PACKAGE_NAME}.jar -C ./classes ${PD_PACKAGE_NAME}
if [ $? -ne 0 ]; then
  exit 1
fi

# Create a manifest file
echo  >jarManifest "Manifest-Version: 1.0"
echo >>jarManifest "Class-Path: ./"
echo >>jarManifest "Main-Class: ${PD_PACKAGE_NAME}.${MAINCLASS}"

# Add the manifest to the archive
"${PD_JDK_BIN_PATH}jar" ufm0 output/${PD_PACKAGE_NAME}.jar jarManifest
if [ $? -ne 0 ]; then
  exit 1
fi

# Success, so delete the manifest file and exit
rm jarManifest
echo "Build successful"

# End
