// **************************************************************************
// * File:   Wrappers.hpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Wrapper classes for non-handle items, ie. basic types (int,	*
// *         bool, char), small-classes, enums, collections and refs.		*
// *         The wrapper provides handle functionality for these types for	*
// *         use in universal unions.										*
// **************************************************************************

#if !defined(_H_WRAPPERS)
#define _H_WRAPPERS 1

//---------------------------------------------------------------------
// Class '_eWrapperBase' - base class for all the wrapper variants ...
//---------------------------------------------------------------------

template <class X > class _eWrapperBase : public _eStorable
{
  protected:
	X Object;
  public :
	_mPerfectTypeInfoHdrNA(_eWrapperBase<X >)		// defines the static function 'myTypeInfo()' and virtual function 'getMyTypeInfo()'

	//-------------
	// Constructors
	//-------------

	_eWrapperBase(const X& x) : _eStorable (), Object(x) {}

	_eWrapperBase(const _aEmptyClass dummy) : _eStorable () {}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	const std::type_info& _lType() const
	{
		return typeid(X);
	}

	X _lValue() const
	{
		return Object;
	}

	X& _lChangeValue()
	{
		return Object;
	}

	// need to define this schema (because it's deferred in the base class, _eStorable), although it will never be called ...
	void _astoreMembers (_eStoreStream *) const
	{
		_mAbsurd;		// should never be called
	}
};

// Function to return an instantiable type-info object that describes the wrapped type ...
template<class X > _eHndl<_eInstblTypeInfo > _eWrapperBase<X >::_aMyTypeInfo()
{
    static _eHndl<_eInstblTypeInfo > ti;
    ti = _atypeInform<X > :: get();
    return ti;
}

//---------------------------------------------------------------------------------------------
// Class '_eStorableWrapperBase' - wrapper for storable small (non-handle) classes, basic types
// (int, bool and char), references (_eRef<X>), storable collections (seq, set, bag, map, pair)
//---------------------------------------------------------------------------------------------
template <class X > class _eStorableWrapperBase : public _eWrapperBase<X >
{
  public:

	//--------------
	// Constructors
	//--------------

	_eStorableWrapperBase(const X& x) : _eWrapperBase<X >(x)
	{
	}

	// Special CTOR used where we need to build an instance on the heap (during de-serialization
	// of a variable that is a union of a collection and something else) ...
	_eStorableWrapperBase(const _aEmptyClass dummy) : _eWrapperBase<X >(dummy)
	{
	}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	_eBool _lEqual(const _eAnyBase *a_wrapper) const
	{
		return a_wrapper != NULL
			&& a_wrapper->_lType() == this->_lType()
			&& this->Object == static_cast<const _eStorableWrapperBase<X >* >(a_wrapper)->Object;
		// Note: the two instances of "this->" in the above are needed for ISO compatibility!
	}

	// Schema to implement functionality for persistant storage
	void _astore(_eStoreStream* stream) const
	{
		// Pass store request on to the global store schema which will in turn call the one in our storage
		// stream (for int, bool, char, collection or user-small-class via the template function)
		_astoreGlobal(stream, this->Object);
	}

	// Schema to implement functionality for persistant storage
	void _afill(_eFillStream* stream)
	{
		// Pass fill request on to the global fill schema which will in turn call the one in our storage
		// stream (for int, bool, char, collection or user-small-class via the template function)
		// Don't use "::" scope resolution here, it upsets gcc 4.3
		_afillGlobal(stream, this->Object);
	}
};

// Wrapper for int, bool, char
template <class X > class _eBasicTypeWrapper : public _eStorableWrapperBase<X >
{
  public:
	//--------------
	// Constructors
	//--------------

	_eBasicTypeWrapper(const X& x) : _eStorableWrapperBase<X >(x) {}

	// Special CTOR used where we need to build an instance on the heap (during de-serialization
	// of a variable that is a union of a collection and something else) ...
	_eBasicTypeWrapper(const _aEmptyClass dummy) : _eStorableWrapperBase<X >(dummy) {}

	_eAnyBase* _lCopy() const
	{
		return new _eBasicTypeWrapper<X >(this->Object);
	}

	_eRank :: _eEnum _oRank(const _eAnyBase* arg) const
	{	const _eBasicTypeWrapper<X>* ptr = dynamic_cast<const _eBasicTypeWrapper<X>*>(arg);
		if (ptr != NULL)
		{
			return ::_oRank(this->Object, ptr->Object);
		}
		return _eAnyBase :: _oRank(arg);
	}

	_eSeq<_eChar> toString() const
	{
		return ::_ltoString(this->Object);
	}
};

// Wrapper for all other storable non-handle classes (seq, set etc)
template <class X > class _eStorableWrapper : public _eStorableWrapperBase<X >
{
  public :
	//--------------
	// Constructors
	//--------------

	_eStorableWrapper(const X& x) : _eStorableWrapperBase<X >(x) {}

	// Special CTOR used where we need to build an instance on the heap (during de-serialization
	// of a variable that is a union of a collection and something else) ...
	_eStorableWrapper(const _aEmptyClass dummy) : _eStorableWrapperBase<X >(dummy) {}

	_eAnyBase* _lCopy() const
	{
		return new _eStorableWrapper<X >(this->Object);
	}

	_eSeq<_eChar> toString() const
	{
		return this->Object.toString();
	}
};

//------------------------------------------------------------------------------------------------------
// Class '_eNonstorableWrapper' - wrapper for non-storable small (non-handle) classes, ie. non-storable
// enums, small-classes, references (_eRef<X>) to and collections of non-storable types ...
//------------------------------------------------------------------------------------------------------

template <class X > class _eNonStorableWrapper : public _eWrapperBase<X >
{
  public :
	//------------
	// Constructor
	//------------

	_eNonStorableWrapper(const X& x) : _eWrapperBase<X >(x) {}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	_eAnyBase* _lCopy() const
	{
		return new _eNonStorableWrapper<X >(this->Object);
	}

	_eBool _lEqual(const _eAnyBase *a_wrapper) const
	{
		return a_wrapper != NULL
			&& a_wrapper->_lType() == this->_lType()
			&& this->Object == static_cast<const _eNonStorableWrapper<X >* >(a_wrapper)->Object;
	}

	// Schema to implement functionality for persistant storage ...
	void _astore(_eStoreStream* stream) const {} 	// This wrapper is not storable, so we don't pass store requests on

	// Schema to implement functionality for persistant storage ...
	void _afill(_eFillStream* stream) {}			// This wrapper is not storable, so we don't pass fill requests on
};

// Wrapper for non-storable enumerations - these are defined via this macros for each enumeration
// NOTE: the code generator MUST emit _mLocalEnumMembers before _mNonStorableEnumWrapper!
#define _mNonStorableEnumWrapper										\
class _eWrapper : public _eNonStorableWrapper<_eEnum>					\
{ public:																\
	_eWrapper(const _eEnum & x) : _eNonStorableWrapper<_eEnum>(x) {}	\
	_eAnyBase* _lCopy() const {	return new _eWrapper(Object); }			\
	_eRank :: _eEnum _oRank(const _eAnyBase* arg) const					\
	{	const _eWrapper* ptr = dynamic_cast<const _eWrapper*>(arg);		\
		return (ptr == NULL) ? _eAnyBase :: _oRank(arg)					\
				: (Object > ptr->Object) ? _eRank :: above				\
					: (Object < ptr->Object) ? _eRank :: below			\
						: _eRank :: same;								\
	}																	\
	_eSeq<_eChar> toString() const { return _ltoString(Object);	}		\
};

// Wrapper for storable enumerations
// NOTE: the code generator MUST emit _mEnumMembers before _mEnumWrapper, otherwise _ltoString will not bind correctly!
#define _mEnumWrapper													\
class _eWrapper : public _eStorableWrapperBase<_eEnum>					\
{ public:																\
	_eWrapper(const _eEnum & x) : _eStorableWrapperBase<_eEnum>(x) {}	\
	_eAnyBase* _lCopy() const { return new _eWrapper(Object); }			\
	_eRank :: _eEnum _oRank(const _eAnyBase* arg) const					\
	{	const _eWrapper* ptr = dynamic_cast<const _eWrapper*>(arg);		\
		return (ptr == NULL) ? _eAnyBase :: _oRank(arg)					\
				: (Object > ptr->Object) ? _eRank :: above				\
					: (Object < ptr->Object) ? _eRank :: below			\
						: _eRank :: same;								\
	}																	\
	_eSeq<_eChar> toString() const { return _ltoString(Object); }		\
};

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
