// **************************************************************************
// * File:   Vectors.hpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of the the vector class								*
// **************************************************************************

// Note (DC, 31-10-2004):
// We now only enable precondition checks and call stack tracing when _dExtraRuntimeChecks is defined nonzero.
// This is because we don't want these methods to appear in the profiling information, because they are called
// very often and the call stack tracing overhead is large enough to distort the profiling figures.
// Also, these classes should only be used by other library classes, which should never violate our preconditions
// as long as their own precondition checks pass; so there is little point in replicating the checks here unless
// we are debugging the library itself.

#if !defined(_hMain_vector)
#define _hMain_vector

//----------------------------------------------------------------
// Class '_eStack' - A stack of integers used by the sort routines
// Note: There is no checking for overflow conditions.
//----------------------------------------------------------------

class _eStack
{
  private:
	_eSize *stack;
	_eSize position;

  public:

	//---------------------------
	// Constructor and destructor
	//---------------------------

	// Basic constructor to initialise stack to given size
	explicit _eStack(_eSize length)
		: stack(new _eSize[length]), position(0)
	{
	}

	// Destructor
	~_eStack()
	{
		delete stack;
	}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	void push(_eSize a)
	{
		stack[position++] = a;
	}

	_eSize pop()
	{
		return stack[--position];
	}

	_eBool isempty()
	{
		return position == 0;
	}
};

//------------------------------------------------------------
// Template class '_eSection'
// This class represents a length and a pointer to a C++ array
//------------------------------------------------------------

template <class X>
class _eSection
{
  public:
	const _eSize nfilled;

	const X* storage;

	//------------
	// Constructor
	//------------

	_eSection(_eSize f, const void* arr)
		: nfilled(f), storage(static_cast<const X*>(arr))
	{
	}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	_eNat filled() const
	{
		return static_cast<_eNat>(nfilled);		// we do this because _eSize translates to C type 'size_t' which is usually
												// an unsigned long and the 'Perfect' code that refers to the 'filledness' of a
												// section compares against an _eNat, giving rise to a comparison between
												// signed and unsigned quantities.
	}

	// Take a slice without copying the data
	_eSection<X > slice(_eSize offset, _eSize len) const
	{
#if _dExtraRuntimeChecks
		_mFunction(slice);
		_mBeginPre _mCheckPre(offset + len <= nfilled, "0,0");
#endif
		return _eSection<X >(len, storage + offset);
	}

	X operator[](int i) const
	{
#if _dExtraRuntimeChecks
		_mOperator([]);
		_mBeginPre _mCheckPre(static_cast<_eSize>(i) < nfilled, "0,0");
#endif
		return storage[i];
	}
};

//--------------------------------------------------------------------------
// Template class '_eVectorBase' - Class used to allocate a new vector. Note
// no destructor or copy constructor (the storage array will be given to a
// vector before any object of this class is destroyed) ...
//--------------------------------------------------------------------------

template <class X>
class _eVectorBase
{
  protected :
	_eSize allocated, filled;
	X* storage;

	//------------
	// Constructor
	//------------

	explicit _eVectorBase(_eSize alloc)
		: filled(0)
	{
		if (alloc == 0)
		{
			allocated = 0;
			storage = NULL;
		}
		else
		{
			allocated = roundUpAlloc(alloc);
			storage = reinterpret_cast<X *>(_eMem::alloc(allocated * sizeof(X)));
		}
	}

	//------------------------------
	// Concrete member functionality
	//------------------------------

	// Determine what allocation size to round up to
	static _eSize roundUpAlloc(const _eSize need)
	{
		return (sizeof(X) >= 4 && need <= 4) ? 4
				: (sizeof(X) >= 2 && need <= 8) ? 8
				: (need + 15) & (~15);  			// round up to next multiple of 16 elements
	}
};

//----------------------------------------------
// Template class '_eVector' - Main vector class
//   ********** Class declaration **********
//----------------------------------------------

template <class X>
class _eVector : public _eVectorBase<X >
{
  private:

	void resize(const _eSize);

  public:

	//----------------------------
	// Constructors and destructor
	//----------------------------

	// Empty vector constructor
	_eVector()
		: _eVectorBase<X >(0)
	{
	}

	// Copy constructor
	_eVector(const _eVector<X > &v);

	// Constructor taking an initial allocation size and a single element
	_eVector(const _eNat itsize, const X &x);

	// Constructor that takes a length for vectors internal array.
	explicit _eVector(const _eNat array_length);

	// Constructor used to build a vector from section
	explicit _eVector(const _eSection<X >);

	// Destructor - takes care of the memory allocated on the heap for storage.
	~_eVector();

	//------------------------------
	// Concrete member functionality
	//------------------------------

	// Equality operator
	_eBool operator==(const _eVector<X >& rhs) const;

	// Copy assignment operator
	_eVector& operator=(const _eVector<X >& rhs);

	const X& operator[](_eNat index) const
	{
#if _dExtraRuntimeChecks
 		_mOperator([]);
		_mBeginPre _mCheckPre(static_cast<_eSize>(index) < this->filled,"0,0");
#endif
		return this->storage[index];
	}

	X& operator[](_eNat index)
	{
#if _dExtraRuntimeChecks
		_mOperator([]);
		_mBeginPre _mCheckPre(static_cast<_eSize>(index) < this->filled,"0,0");
#endif
		return this->storage[index];
	}

	const X& head() const
	{
#if _dExtraRuntimeChecks
		_mSelector(head);
		_mBeginPre _mCheckPre(this->filled != 0, "0,0");
#endif
		return this->storage[0];
	}

	X& _shead()
	{
#if _dExtraRuntimeChecks
		_mSelector(head);
		_mBeginPre _mCheckPre(this->filled != 0, "0,0");
#endif
		return this->storage[0];
	}

	const X& last() const
	{
#if _dExtraRuntimeChecks
		_mSelector(last);
		_mBeginPre _mCheckPre(this->filled != 0, "0,0");
#endif
		return this->storage[this->filled - 1];
	}

	X& _slast()
	{
#if _dExtraRuntimeChecks
		_mSelector(last);
		_mBeginPre _mCheckPre(this->filled != 0, "0,0");
#endif
		return this->storage[this->filled - 1];
	}

	_eNat _oHash() const
	{
		return static_cast<_eNat>(this->filled);
	}

	_eNat freeCells() const
	{
		return static_cast<_eNat>(this->allocated - this->filled);
	}

	const _eSection<X > getSection() const
	{
		return _eSection<X >(this->filled, this->storage);
	}

	void insert(_eNat index, const X& x);

	void insertSlice(const _eSection<X > section, _eNat destPos, _eNat srcPos, _eNat sizeOfSlice);

	void _rdelete(_eSize first, _eSize number);

	// Add to a vector (inline for efficiency of for..yield constructs)
	void add(const X& x)
	{
		if (this->filled == this->allocated)
		{
			resize(this->filled + 1);
		}
		this->storage[this->filled++] = x;
	}

	// Swap a pair of elements - we do blind copying to avoid unnecessary adjustment of use counts.
	void swapelem(const _eSize a, const _eSize b)
	{
		_lswap(this->storage[a], this->storage[b]);
	}
};

//----------------------------------------------
// Template class '_eVector' - Main vector class
//   ********** Member definitions **********
//----------------------------------------------

// Copy constructor
template <class X> _eVector<X > :: _eVector(const _eVector<X > &v)
	: _eVectorBase<X >(v.filled)
{
	for (_eSize i = 0; i < v.filled; ++i)
	{
		this->storage[i] = v.storage[i];
	}
	this->filled = v.filled;
}

// Constructor taking an initial allocation size and a single element
template <class X> _eVector<X > :: _eVector(const _eNat itsize, const X &x)
	: _eVectorBase<X >(itsize)
{
	add(x);
}

// Constructor used by Variants.cpp
template<class X> _eVector<X > :: _eVector(const _eNat initial_size)
	: _eVectorBase<X >(initial_size)
{
}

// Constructor used to build a vector from a section
template<class X> _eVector<X > :: _eVector(const _eSection<X > s)
	: _eVectorBase<X >(s.nfilled)
{
	for (_eSize i = 0; i < s.nfilled; ++i)
	{
		this->storage[i] = s.storage[i];
	}
	this->filled = s.nfilled;
}

// Destructor - calls all the destructors in the array and then destroys the array.
template<class X> _eVector<X >::~_eVector()
{
	for (_eSize i = 0; i < this->filled; i++)
	{
		this->storage[i].~X();
	}
    _eMem::free(this->storage, this->allocated * sizeof(X));
}

// Insert a single element
template<class X> void _eVector<X >::insert(_eNat index, const X& x)
{
#if _dExtraRuntimeChecks
	_mSchema(insert);
    _mBeginPre _mCheckPre(static_cast<unsigned int>(index) <= this->filled, "0,0");
#endif

	if (this->filled == this->allocated)				        // if we need a bigger vector
	{
		X* oldStorage = this->storage;							// save the old storage
		this->storage = NULL;
		resize(this->filled + 1);								// allocate more storage
		memcpy(this->storage, oldStorage, index * sizeof(X)); 	// copy elements before the insert point
		memcpy(&(this->storage[index + 1]), &oldStorage[index], (this->filled - index) * sizeof(X));
																// copy elements after the insert point
		_eMem :: free(oldStorage, this->filled * sizeof(X));	// "filled" is same as original "allocated"
	}
	else
	{
		// memmove the memory.this is safe with handles because no use count change is needed.
		memmove(&(this->storage[index+1]), &(this->storage[index]), (this->filled - index) * sizeof(X));

		// zero the array element, without affecting the object that was just moved,
		// so that we can assign the new element without affecting the use count of the old
		memset(&(this->storage[index]), 0, sizeof(X));
	}

	// insert the new member in the array.
    this->storage[index] = x;
    ++(this->filled);
}

// Insert a section of one vector (from srcPos for sizeofslice elements) into another vector at destPos
template<class X> void _eVector<X >::insertSlice(const _eSection<X > section, _eNat destPos, _eNat srcPos, _eNat sizeOfSlice)
{
#if _dExtraRuntimeChecks
	_mSchema(insertSlice);
	_mBeginPre _mCheckPre(static_cast<unsigned int>(destPos) <= this->filled, "0,0");
#endif

    if (sizeOfSlice != 0)
	{
		if (this->filled + sizeOfSlice > this->allocated)			// if we need a bigger vector
		{
			X* oldStorage = this->storage;							// save the old storage
			const _eSize oldSize = this->allocated * sizeof(X);  	// save the old size
			this->storage = NULL;
			resize(this->filled + sizeOfSlice);						// allocate more storage
			memcpy(this->storage, oldStorage, destPos * sizeof(X));	// copy elements before the insert point
			memcpy(&(this->storage[destPos + sizeOfSlice]), &oldStorage[destPos], (this->filled - destPos) * sizeof(X));
																	// copy elements after the insert point
			_eMem :: free(oldStorage, oldSize);						// free the old memory
		}
		else
		{
			// memmove the memory. This is safe with handles because no use count change is needed.
			memmove(&(this->storage[destPos + sizeOfSlice]), &(this->storage[destPos]), (this->filled - destPos) * sizeof(X));

			// zero the array element, without affecting the object that was just moved,
			// so that we can assign the new element without affecting the use count of the old
			memset(&(this->storage[destPos]), 0, sizeOfSlice * sizeof(X));
		}

    	for (_eNat i = 0; i < sizeOfSlice; i++)
			this->storage[destPos + i] = section.storage[srcPos + i];
																	// copy the slice into the gap
    	this->filled += sizeOfSlice;								// update the allocation.
	}
}

// Assignment operator - We must allow for the case where we are assigning a vector to itself!
// (i.e. either check for this case, or don't delete old value before assigning new). However,
// since different vectors cannot share the same array, we need not consider assigning a slice
// of a vector to itself.
//??? currently, this always allocates new storage - we should optimise it!
template<class X> _eVector<X >& _eVector<X > :: operator=(const _eVector<X >& x)
{
	if (&x != this)				                    // shortcut if null assignment
	{
		X* new_pointer;
		if (x.filled == 0)							// avoid allocating zero length array
		{
			new_pointer = NULL;
		}
		else
		{
			new_pointer = static_cast<X*>(_eMem::alloc(x.filled * sizeof(X)));
   			for (_eSize j = 0; j < x.filled; j++)
			{
				new_pointer[j] = x[j];
			}
		}
		this -> ~_eVector<X >();				        // destroy the old storage and free it
		this->storage = new_pointer;
		this->filled = this->allocated = x.filled;
	}
   	return *this;
}

// Change the allocated size
// NOTE: this may be called with the vector in an abnormal state with "storage == NULL & filled > 0"
// In this case, we must not attempt to copy the storage, nor must we tamper with "filled".
template<class X> void _eVector<X >::resize(const _eSize size)
{
#if _dExtraRuntimeChecks
	_mSchema(resize);
	_mBeginPre _mCheckPre(size >= this->filled, "0,0");
#endif

	const _eSize newSize = this->roundUpAlloc(size);
	X* new_pointer = static_cast<X*>(_eMem::alloc(newSize * sizeof(X)));

	if (this->storage != NULL)
	{
		memcpy(new_pointer, this->storage, this->filled * sizeof(X));	// copy the old memory to the new.
		_eMem::free(this->storage, this->allocated * sizeof(X));		// free the old memory.
	}
	this->storage = new_pointer;										// let the pointer look at the new memory.
	this->allocated = newSize;
}

// Delete a slice of the current value
template <class X> void _eVector<X >::_rdelete(_eSize first, _eSize number)
{
#if _dExtraRuntimeChecks
	_mSchema(_rdelete);
	_mBeginPre _mCheckPre(first + number <= this->filled, "0,0");
#endif

	if (number != 0)
	{
		//delete the slice we are removing.
    	for (_eSize i = first; i < first + number; i++)
            (this->storage + i)->~X();

		//compress the array to fill the gap.
		memmove(&(this->storage[first]), &(this->storage[first + number]), (this->filled - (first + number)) * sizeof(X));

		//adjust filled to represent the new array.
    	this->filled -= number;

		//zero the end of the list
		memset(&(this->storage[this->filled]), 0, sizeof(X)*number);
	}
}

// Equality operator (ignores "allocated" field)
template<class X> bool _eVector<X >::operator==(const _eVector<X >& v) const
{
    if (this->filled != v.filled)
		return false;

    for (_eSize i = 0; i < this->filled; i++)
	{
        if (!(operator[](i) == v[i]))
        	return false;
	}
    return true;
}

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
