// **************************************************************************
// * File:   Heaps.hpp														*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Heap functionality.										*
// *         Anything that is on the heap is there because something else	*
// *         refers to it. This means it cannot be deleted, i.e.			*
// *         !CanDelete() is an invariant. This is true even at the end of	*
// *         the program, however since it is the end, we can 'Destroy' it	*
// *         anyway. Note that the act of 'destroying' may or may not lead	*
// *         to the link's (and other links') deletion, depending on		*
// *         whether the object refers to itself (or other links). Note		*
// *         also that although there is no point in destroying a link		*
// *         twice, it does no harm. Thus deletion simply cycles through,	*
// *         destroying undestroyed links until they've all gone.			*
// **************************************************************************

#if ! defined (_HPP_HEAPS)
#define _HPP_HEAPS

#include "StorableSupport_1.hpp"		// for 'RefTypeInfo()'

//--------------------------------
// Forward delcarations of classes
//--------------------------------

class _eHeap;
class _eHeapLink;

namespace _eEscherRuntime
{
	extern _eNat getNextSerialNumber();
}

//---------------------------------------------------------
//                Class '_eHeapElemBase'
// The base class for links in the heap. Note that we treat
// '_eHeap' as a derived class as this is neater.
//---------------------------------------------------------

class _eHeapElemBase
{
  public :
	//-----------
	// Destructor
	//-----------

	virtual ~_eHeapElemBase()
	{
	}

	// used by function 'getHeap(..)' in _eRef<X> to find the target heap from an _eHeapElement
	virtual _eBool isHeap() const
	{
		return false;
	}

  protected :
	//----------------------
	// Protected constructor
	//----------------------

	_eHeapElemBase() : Next(NULL)
	{
	}

	//-----------------
	// Access modifiers
	//-----------------

	friend class _eHeapLink;
	friend class _eHeap;

	//------------------------------------------------------------
	// Protected data - the link the next heap element (accessable
	// to classes '_eHeapLink' and '_eHeap')
	//------------------------------------------------------------

	_eHeapLink* Next;
};

//-------------------
// Class '_eHeapLink'
//-------------------

class _eHeapLink : public _eHeapElemBase
{
  private :

    // we use this pointer to maintain the linked list
	_eHeapElemBase* Previous;

	// we use this as the reference count
	_eUseCount UseCount;

  protected :
	bool IsDestroyedFlag;

  public :

	// used by function 'getHeap(..)' in _eRef<X> to find the target heap from an _eHeapElement
	const _eHeapElemBase* getPrevious() const
	{
		return Previous;
	}

	//----------------------
	// Default CTOR and DTOR
	//----------------------

	_eHeapLink(_eHeap* heap);
	//
	~_eHeapLink();

	//-----------------------------
	// reference counting functions
	//-----------------------------

	void IncUseCount()
	{
		++UseCount;
	}
	//
	void DecUseCount()
	{
		_mCannotHappen(UseCount == 0, "Use count already zero");
		--UseCount;
	}

	//----------------------------
	// link destruction management
	//----------------------------

	bool CanDelete() const
	{
		return UseCount == 0 && !IsDestroyedFlag;
	}
	//
  protected :
	//
	friend class _eHeap;
	//
	bool IsDestroyed() const
	{
		return IsDestroyedFlag;
	}
	//
	virtual void Destroy() = 0;
};

//---------------
// Class '_eHeap'
//---------------

class _eHeap : public _eHeapElemBase
{
  public :

	//----------------------
	// Default CTOR and DTOR
	//----------------------

	// used by function 'getHeap(..)' in _eRef<X> to find the target heap from an _eHeapElement
	_eBool isHeap() const
	{
		return true;
	}

	_eHeap() : _eHeapElemBase()
	{
	}
	//
	~_eHeap();

};

//--------------------------------------------------------------------------------
//                        Templated class '_eRef' - DECLARATION
// This class defines the reference to an object that we want to store on the heap
// A list of references on a heap is a doubly-linked-list where the 'Next' thing
// on the heap is the last thing added, and where the 'Previous' thing as far as
// the last thing added is concerned, is the heap itself.
//--------------------------------------------------------------------------------

template<class X> class _eRef
{
  public :
    _mNew (_eRef)			// we provide this so that references get allocated from the freelists
	_mPerfectTypeInfoHdrNA(_eRef<X >)	// just defines the type-info obtaining functions
	_mPerfectTypeInfoFwd	// defines the _aGetMyTypeInfo() forwarding function (used to be part of the xxxHDRxx macro)

  #if !defined(__INTEL_COMPILER) && !defined(__ghs__)
	private :				// if using the Intel compiler _eHeapElem must be declared public for its own delete operator to compile
  #endif

	//---------------------------------------------------------------------------------
	//                              Class '_eHeapElem'
	// The mechanism implemented is such that a '_eHeapElem' instance contains the
	// object we want to store on the chosen heap and a '_eHeapLink' instance to link
	// it to both the next element and back to the heap itself. A heap-link also holds
	// a use count similar to handles, for memory management purposes.
	//---------------------------------------------------------------------------------

	class _eHeapElem : public _eHeapLink
	{
	  public :
    	_mNew (_eHeapElem)			// we provide this so that the heap-element pointed to by a reference gets allocated from a freelist

		//-----
		// Data
		//-----

		X Object;					// the thing we want to store on the heap

	  private:

		_eNat serialNumber;			// key - used for ordering

	  public:

		//-------------------------------------------
		// CTOR taking our object and the chosen heap
		//-------------------------------------------

		_eHeapElem(const X& x, _eHeap *Heap)
			: _eHeapLink(Heap), Object(x)
		{
			serialNumber = _eEscherRuntime::getNextSerialNumber();
		}

	  protected :
		//--------------------------------------------------
		// Function to destroy the object held by this heap
		// element. Note that this does not destroy the
		// reference to it ...
		//--------------------------------------------------

		void Destroy()
		{
			IsDestroyedFlag = true;
			Object.~X();
		}

	  public :
		//----------------------------------------------------------
		// Functionality required by the de-serialization mechanism
		//----------------------------------------------------------

		_eHeapElem(_eHeap *Heap, _eNat serialNum)
			: _eHeapLink(Heap), serialNumber(serialNum)
		{
		}

		_eNat getSerialNumber() const
		{
			return serialNumber;
		}
	};

  private :

	//-------------------------------------------
	// Data - a reference to an object on a heap
	//-------------------------------------------

	_eHeapElem* Element;

	// recursive function to find the address of the heap on which this references resides (saves us storing redundant data) ...
	const _eHeapElemBase* getHeap(const _eHeapElemBase* elem) const
	{
		if(elem -> isHeap() != true)
			return getHeap(static_cast<const _eHeapLink*>(elem) -> getPrevious());	// recurse backwards up the list ...
		else
			return elem;															// reached the heap itself, OK.
	}

  public :

	_eAddress heapAddr;				// public data - the actual heap address (used by the serialization mechanism only)

	// dummy toString function - can't call toString on a ref, so this is just defined to satisfy the compiler
	// in the case where we have a wrapped _eRef<X>, and as a result we make it absurd ...
	_eSeq<_eChar> toString() const
	{
		_mAbsurd;
		return _mString("");
	}

	_eRank :: _eEnum _oRank (const _eRef other) const
	{
    	_mOperator (~~);
		const _eNat thisSerial = Element -> getSerialNumber();
		const _eNat otherSerial = other.Element -> getSerialNumber();
    	return ((thisSerial > otherSerial) ? _eRank :: above : (thisSerial < otherSerial) ? _eRank :: below : _eRank :: same);
	}

	//--------------------------------------
	// CTORs and DTORs for the '_eRef' class
	//--------------------------------------

	// Destructor
	~_eRef();

	// Default constructor
	_eRef() : Element(NULL)
	{
		heapAddr = NULL;
	}

	// Copy constructor
	_eRef(const _eRef&);

	// Normal constructor, taking the heap we want to use and
	// the object we want to store on the heap
    _eRef(const X& x, _eHeap* Heap)
		: Element(new _eHeapElem(x, Heap))
	{
		heapAddr = static_cast<_eAddress>(Heap);
	}

	//---------------------------------
	// Asignment and equality operators
	//---------------------------------

	_eRef& operator=(const _eRef& ref);

	_eBool operator==(const _eRef& other) const
	{
		return Element == other.Element;
	}

	//-------------------------------------------------
	// Selector to get at the object referenced by this
	// '_eRef' instance
	//-------------------------------------------------

	const X& value() const
	{
	   	_mSelector(value);
      	_mBeginPre _mCheckPre(Element != NULL, "0,0");
		return Element -> Object;
	}

	X& value()
	{
	   	_mSelector(value);
      	_mBeginPre _mCheckPre(Element != NULL, "0,0");
		return Element -> Object;
	}

	//----------------------------
	// Null checking functionality
	//----------------------------

	bool IsNull() const
    {
    	return Element == NULL;
    }

	bool IsNotNull() const
    {
    	return Element != NULL;
    }

	const _eRef& CheckNotNull(const _eNativeChar* file, const int line) const
	{
		if (Element != NULL)
		{
			return *this;
		}
		throw _xCannotHappen(_mCstr("Reference was not expected to be NULL"), file, line);
	}

	_eRef& CheckNotNull(const _eNativeChar* file, const int line)
	{
		if (Element != NULL)
		{
			return *this;
		}
		throw _xCannotHappen(_mCstr("Reference was not expected to be NULL"), file, line);
	}

	void* CheckIsNull(const _eNativeChar* file, const int line)
	{
		if (Element == NULL)
		{
			return NULL;
		}
		throw _xCannotHappen(_mCstr("Reference was expected to be NULL"), file, line);
	}

	//------------------------------------------------------------------
	// Routines to implement persistant storage for ref's ...
	//------------------------------------------------------------------

	// This is the constructor we use when we build the first occurance of a ref during de-serialization, ie. when we saw
	// code CV_REF in the stream ...
	_eRef(const _eNat serialNumber, const _eAddress targetHeap)
	{
		_eHeap* heap = const_cast<_eHeap*>(static_cast<const _eHeap*>(targetHeap));
		Element = new _eHeapElem(heap, serialNumber);
		heapAddr = targetHeap;
	}

	// This is the constructor we use when we build an occurance of a ref we've already seen during de-serialization, ie. when
	// we saw code CV_ANOTHERREF in the stream. Should NEVER be called with a null-address ...
	_eRef(const _eAddress addr)
	{
		_mBuild;
		_mBeginPre _mCheckPre(addr != NULL, "0,0");
    	Element = const_cast<_eHeapElem*>(static_cast<const _eHeapElem*>(addr));
		Element -> IncUseCount();
		heapAddr = static_cast<_eAddress>(getHeap(Element));
	}

	// return the size of the _eHeapElem object that this reference refers to ...
	_eSize memSize() const
	{
		return sizeof *Element;
	}

	// return the address of the actual object this reference refers to ...
	_eAddress _rthis() const
	{
		return static_cast<_eAddress>(Element);
	}

	// return the serial number of the element referred to by this ref object ...
	_eNat getSerialNumber() const
	{
		_mFunction(getSerialNumber);
		_mBeginPre _mCheckPre(Element != NULL, "0,0");
		return Element -> getSerialNumber();
	}

	// This store schema is not used for the case where we store a data member of type _eRef<X>, since the code generation
	// optimizes this by calling the _astoreRef(..) schema in the storage stream class directly. For the other cases (ie.
	// where we have a collection of ref's; a template class instantiated with ref and that has a data member of that type;
	// or a union involving ref) we forward the store request on to the storage-stream, passing the address of the target heap
	// so that the stream can find a heap index (the stream keeps track of all heap names via HeapInfo objects, and the addresses
	// of the corrosponding heap; note that the heap is guaranteed to have been registered at this point because the '_astoreMembers'
	// schema makes calls to the heap registering schema in the stream for all heaps involved in the union). The storage stream
	// then call the '_adoStore(..) schema (see below) to actually store the reference object
	void _astore(_eStoreStream* stream) const;

	// This fill schema is only used when we fill a universal union of a ref and something non-void. This ref will have been
	// created on the heap via the instantiation table / instantiate function system (except in the case of a union of a ref
	// with void, when this ref instance will have been created by the default ref constructor simply because of the declaration
	// in the target class that we're filling) ...
	void _afill(_eFillStream* stream);

	// This store schema actually stores the referenced object in the stream. It is called from the _astoreRef(..) schema in
	// the storage stream class
	void _adoStore(_eStoreStream* stream) const;

	// This fill schema actually fills the referenced object from the stream. It is called from the _afillRef(..) schema in
	// the storage stream class
	void _adoFill(_eFillStream* stream);
};

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
