// **************************************************************************
// * File:   Variants.hpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Classes, exceptions, functions and macros used to check		*
// *         variants.														*
// **************************************************************************

#if !defined(_h_Variants)
#define _h_Variants	(1)

//-------------------------------
// Variant base class '_eVariant'
//-------------------------------

#if !defined(NDEBUG)

  typedef int _eVar;

  class _xVariant;	// Forward declaration since _eVariant is a friend of _xVariant

  class _eVariant
  {
    public:

	  //------------
	  // Constructor
	  //------------

	  _eVariant(bool is_checking)
	  	: Index(0), IsChecking(is_checking)
	  {
	  }

	  //-------------------------------
	  // Concrete member functionality
	  //-------------------------------

	  bool Check(_eVar x);

	  void CannotEvaluate();

    protected:

	  //----------------------
	  // Protected member data
	  //----------------------

	  _eSize Index;

	  //----------------------------------------
	  // Concrete protected member functionality
	  //----------------------------------------

	  friend class _xVariant;

	  bool IsChecking;

	  //----------------------------------------
	  // Abstract protected member functionality
	  //----------------------------------------

	  virtual _eVar& This(_eSize i) = 0;

	  virtual _eVar This(_eSize i) const = 0;

	  virtual _eVar Last(_eSize i) const = 0;

	  virtual _eSize Number() const = 0;

	  virtual void CheckEnd() = 0;
  };

  class _xVariant : public _xDebug
  {
    protected:

	  _eVector<_eVar> List;
	  _eVar Tried;
	  _eSize Index;

	  void appendMessage(_eWorkspace ws, _eSize& size) const;

    public:

	  _xVariant(const _xVariant& x)
		  : _xDebug(x), List(x.List), Tried(x.Tried), Index(x.Index)
	  {
	  }

	  _xVariant(_eVar tried, const _eVariant& v, const _eCallStackInfo& info, _eCstr file_name, _eNat line);
  };

#endif

//-------------------
// Loop variant class
//-------------------

#if !defined(NDEBUG)

  template<_eSize N>
  class _eLpVariant : public _eVariant
  {
    public:

	  _eLpVariant()
	  	  : _eVariant(false)
	  {
	  }

    protected:

	  //----------------------
	  // Protected member data
	  //----------------------

	  _eVar List[N];

	  //-------------------------------
	  // Protected member functionality
	  //-------------------------------

	  _eVar Last(_eSize i) const
      {
		  _mCannotHappen(i >= N, "No such variant");
		  return List[i];
	  }

	  _eVar& This(_eSize i)
	  {
		  _mCannotHappen(i >= Index, "That variant has not been checked");
		  return List[i];
	  }

	  _eVar This(_eSize i) const
      {
		  _mCannotHappen(i >= Index, "That variant has not been checked");
		  return List[i];
	  }

	  _eSize Number() const
	  {
	  	  return N;
	  }

	  void CheckEnd()
	  {
		  if (Index == N)
		  {
			IsChecking = true;
			Index = 0;
		  }
	  }

  };

#endif

//----------------------------------
// Loop variant exception definition
//----------------------------------

#if !defined(NDEBUG)

  class _xLoopVariant : public _xVariant
  {
    public:

	  _xLoopVariant(
                _eVar tried,
                const _eVariant& v,
                const _eCallStackInfo& info,
                _eCstr file_name,
                _eNat line) : _xVariant(tried, v, info, file_name, line)
	  {
	  }

    protected:

	  void appendHeader(_eWorkspace ws, _eSize& size) const
      {
		  append(ws, _mCstr("Loop Variant failed\n"), size);
	  }

  };

  #define _mLoopVariant(N)		_eLpVariant<N> _tVariant;

  #define _mBeginLoopDecrease \
	if (_eEscherRuntime::EnableLoopDecrease && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
	{ ++_eEscherRuntime::CurrentCheckNesting; \
	  try {

  #define _mCheckLoopDecrease(v, position) \
	  if (!_tVariant.Check(v)) throw _xLoopVariant(v, _tVariant, _eCallStackInfo(_tSelf.functionname, position), _mCurrentFile, __LINE__);	\
	  } catch (_xCannotEvaluate) { _tVariant.CannotEvaluate(); } \
	  --_eEscherRuntime::CurrentCheckNesting;	\
	}

#else

  #define _mLoopVariant(N)						_mNoact
  #define _mBeginLoopDecrease					_mNoact
  #define _mCheckLoopDecrease(next, position)	_mNoact

#endif

//-----------------------
// Function variant class
//-----------------------

#if !defined(NDEBUG)

  template<_eSize N>
  class _eFnVariant : public _eVariant
  {
    public:
	  _eFnVariant(_eFnVariant*& ptr)
		: _eVariant(ptr != NULL), Pointer(&ptr), LastVariant(ptr)
	  {
		  *Pointer = this;
	  }

	  ~_eFnVariant()
	  {
		  *Pointer = LastVariant;
	  }

    protected:

	  //----------------------
	  // Protected member data
	  //----------------------

	  _eFnVariant** Pointer;
	  _eFnVariant* LastVariant;
	  _eVar List[N];

	  //-------------------------------
	  // Protected member functionality
	  //-------------------------------

	  _eVar& This(_eSize i)
	  {
		  _mCannotHappen(i >= Index, "That variant has not been checked");
		  return List[i];
	  }

	  _eVar This(_eSize i) const
	  {
		  _mCannotHappen(i >= Index, "That variant has not been checked");
		  return List[i];
	  }

	  _eVar Last(_eSize i) const
	  {
		  _mCannotHappen(i >= N, "No such variant");
		  return LastVariant->List[i];
	  }

	  _eSize Number() const
	  {
	  	  return N;
	  }

	  void CheckEnd()
	  {
	  }
  };

#endif

//----------------------------
// Specification variant class
//----------------------------

#if !defined(NDEBUG)

  class _xSpecVariant : public _xVariant
  {
    public:
	  _xSpecVariant(
            _eVar tried,
            const _eVariant& v,
            const _eCallStackInfo& info,
            _eCstr file_name,
            _eNat line) : _xVariant(tried, v, info, file_name, line)
	  {
	  }

    protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Recursive Function Specification Variant failed\n"), size);
	  }
  };

  #define _mSpecRecursion(N) \
	static _eFnVariant<N>* _tSpecStack = NULL; \
	_eFnVariant<N> _tSpecVariant(_tSpecStack);

  #define _mBeginSpecDecrease \
	if (_eEscherRuntime::EnableSpecDecrease && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting) \
	{ ++_eEscherRuntime::CurrentCheckNesting; \
	  try {

  #define _mCheckSpecDecrease(v, position) \
	  if (!_tSpecVariant.Check(v)) \
	    throw _xSpecVariant(v, _tSpecVariant, _eCallStackInfo(_tSelf.functionname, position), _mCurrentFile, __LINE__);	\
	  } catch (_xCannotEvaluate) { _tSpecVariant.CannotEvaluate(); } \
	  --_eEscherRuntime::CurrentCheckNesting;	\
	}

#else

  #define _mSpecRecursion(N) 				_mNoact
  #define _mBeginSpecDecrease				_mNoact
  #define _mCheckSpecDecrease(v, position)	_mNoact

#endif

//-----------------------------
// Implementation variant class
//-----------------------------

#if !defined(NDEBUG)

  class _xImpVariant : public _xVariant
  {
    public:
	  _xImpVariant(
            _eVar tried,
            const _eVariant& v,
            const _eCallStackInfo& info,
            _eCstr file_name,
            _eNat line) : _xVariant(tried, v, info, file_name, line)
	  {
	  }

    protected:
	  void appendHeader(_eWorkspace ws, _eSize& size) const
	  {
		  append(ws, _mCstr("Recursive Function Implementation Variant failed\n"), size);
	  }
  };

  #define _mImpRecursion(N) \
	static _eFnVariant<N>* _TimpStack = NULL; \
	_eFnVariant<N> _tImpVariant(_TimpStack);

  #define _mBeginImpDecrease \
	if (_eEscherRuntime::EnableImpDecrease && _eEscherRuntime::CurrentCheckNesting < _eEscherRuntime::MaxCheckNesting)	\
	{ ++_eEscherRuntime::CurrentCheckNesting; \
	  try {

  #define _mCheckImpDecrease(v, position) \
	  if (!_tImpVariant.Check(v)) \
	  	throw _xImpVariant(v, _tImpVariant, _eCallStackInfo(_tSelf.functionname, position), _mCurrentFile, __LINE__);	\
	  } catch (_xCannotEvaluate) { _tImpVariant.CannotEvaluate(); } \
	  --_eEscherRuntime::CurrentCheckNesting;	\
	}

#else

  #define _mImpRecursion(N)					_mNoact
  #define _mBeginImpDecrease				_mNoact
  #define _mCheckImpDecrease(v, position)	_mNoact

#endif

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
