// **************************************************************************
// * File:   String.hpp														*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definitions for string class.									*
// **************************************************************************

#if !defined(_h_String)
#define _h_String

//-------------------------------------------
// Definition of string as a sequence of char
//-------------------------------------------

typedef _eSeq<_eChar> _rstring;

//----------------------------------------------------------------------
// Function to convert a C null-terminated string to a 'Perfect' string.
//----------------------------------------------------------------------

extern _rstring _lString(const _eNativeChar *);

//------------------------------------------------------------------------
// _mString macro define in 'MiscMacs.hpp' is more efficient than _lString
// for converting string literals.
//------------------------------------------------------------------------

//------------------------------------------------------------------------
// Functions to convert basic types to 'Perfect' strings.
// Note: When 'toString' is called on a value 'x' of a template parameter type,
// the compiler generates a call '::_ltoString(x)' because X might be of a type
// such as int, char etc. for which no toString member can be defined.
// This means that ::_ltoString(x) must be defined for all types.
// We define it here for basic types, handles and from's.
// We define it in union_1.hpp for unions.
// For enumerations, it is defined by macro _mEnumMembers and access to it
// is provided by macro _mEnumGlobalSupport.
//------------------------------------------------------------------------

extern _rstring _ltoString(const _eInt i);
extern _rstring _ltoString(const _eReal r);
extern _rstring _ltoString(const _eByte b);
extern _rstring _ltoString(const _eChar c);
extern _rstring _ltoString(const _eBool b);

// Global functions to support toString in template methods
template <class X> inline _rstring _ltoString(_eHndl<X> arg)
{
	return (arg.Ptr() == NULL ? _mString("null") : arg -> toString());
}

template <class X> inline _rstring _ltoString(_eFrom<X> arg)
{
	return (arg.Ptr() == NULL ? _mString("null") : arg -> toString());
}

// Other cases.
// We could avoid these by generating a global _ltoString function for any small classes that define 'toString'.
template <class X> inline _rstring _ltoString(_eSet<X> arg)
{
	return arg.toString();
}
template <class X> inline _rstring _ltoString(_eBag<X> arg)
{
	return arg.toString();
}
template <class X> inline _rstring _ltoString(_eSeq<X> arg)
{
	return arg.toString();
}
template <class X, class Y> inline _rstring _ltoString(_ePair<X, Y> arg)
{
	return arg.toString();
}
template <class X, class Y, class Z> inline _rstring _ltoString(_eTriple<X, Y, Z> arg)
{
	return arg.toString();
}

//------------------------------------------------------
// Function to convert a 'Perfect' string to an integer.
//------------------------------------------------------

extern _eNat _lstringToNat(const _rstring s);

//-----------------------------------------------------------------------------
// Class '_eCstring' - used when we convert a 'Perfect' string to a C string.
// The purpose of this is to track the size of the temporary buffer we allocate
//-----------------------------------------------------------------------------

class _eCstring
{
  private:
	_eNativeChar *s;									// pointer to heap buffer holding the string
	_eSize length;										// length of the buffer

	_eCstring& operator=(const _eCstring&);				// declare a private assignment operator to prevent assignment

  public:
	//----------------------------
	// Constructors and destructor
	//----------------------------

	_eCstring(const _rstring&);			                // build a C string from a 'Perfect' string
	_eCstring(const _eNativeChar*);				        // build a C string from a C string literal or similar
	_eCstring(_eSize size);								// build a C string with an empty buffer
	_eCstring(const _eCstring&);			            // copy constructor

	~_eCstring();									    // destroy the object, releasing the memory

	//----------------------
	// General functionality
	//----------------------

	// Return the buffer length
	size_t bufLength() const
	{
		return length;
	}

	// Return a constant C string
	const _eNativeChar* str() const
	{
		return s;
	}

	// Return a changeable C string
	_eNativeChar* changeStr()
	{
		return s;
	}

	// Return the strlen of the string
	size_t strlen() const;

	// assign an instance to this one - on purpose!
	// WARNING: This doesn't allocate any memory, but just assigns pointer to that of the string we're
	//          assigning to - be VERY careful if using this call!!!
	void assign(const _eCstring &other)
	{
		_eMem::free(s, length * sizeof(_eChar));
		s = other.s;
		length = other.length;
	}

	// Return the equivalent 'Perfect' string
	_rstring eselString() const;
};

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
