// **************************************************************************
// * File:   ClassMacros.hpp												*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definitions of macros used in generated classes to pull-in		*
// *         runtime functions appropriate to the type of class				*
// *         (abstract/concrete and direct/indirect).						*
// **************************************************************************

#if !defined(_h_ClassMacros)
#define _h_ClassMacros 1

// Define equality with other objects
#define _mEqualsOperator(X) 											\
	_eBool _lEqual(const _eAnyBase *h) const							\
	{																	\
		return     this == h											\
				|| (   h != NULL										\
					&& typeid(X) == h->_lType()							\
					&& *this == static_cast<const X*>(h)				\
			   	   );													\
	}

// Define a function to generate a copy of an object
#define _mDefineCopy( X )												\
	_eAnyBase* _lCopy() const											\
	{																	\
		return new X(*this);											\
	}

// Define a function to return the type of an object
#define _mDefineType( X )												\
	const std::type_info& _lType() const										\
	{																	\
		return typeid( X );												\
	}

// Define memory management for the object. This can be changed
// to maintain a freelist for each type of object
#define _mNew( X )														\
	static void* operator new(size_t sz)								\
	{																	\
		return _eMem::alloc(sz);										\
	}																	\
	\
	static void operator delete(void* ptr)								\
	{																	\
		_eMem::free(ptr, sizeof(X));									\
	}

// Define memory management for the object for the case where we want to
// use placement syntax (eg. during de-serialization of references).
#define _mPlacementNew( X )												\
	static void* operator new(size_t sz, void* preAllocMem)				\
	{																	\
		return preAllocMem;												\
	}																	\
	\
	static void operator delete(void* ptr, void*)						\
	{																	\
		_eMem::free(ptr, sizeof(X));									\
	}

// Define a static build function corresponding to the default constructor
// Due to a bug in Microsoft VC++ 6.0 we must not include an empty parameter list after "new X"
#define _mDefineBuild0(X)												\
	static _eHndl<X > build()											\
	{																	\
		static bool _vDone = false;										\
		static _eHndl<X > _vRslt;										\
		if (!_vDone)       												\
		{																\
			_vRslt = new X;												\
			_vDone = true;												\
		}																\
		return _vRslt;													\
	}

// Macro used at the start of a constructor for a handle class
// The purpose is to increment the use count of the current object, preventing it from
// deletion if a handle is created from the current object and then destroyed
// We do this by creating a object that increments the use count of 'this'
// when it is created and decrements it (without releasing the object)
// when it is destroyed at the end of the constructor
#define _mConstructor													\
	_eConstructorHandle _lKeepSelf(this);

// Macros used for all perfect classes for defining the special functions for returning type information about
// a class. The static one is used by the 'myTypeInfo()' functions in template classes, when defining the template
// arguments, and the other one is used for unions, so that we can obtain the type-info of the live union at
// runtime (the function is pure-virtual in the base class). Note that we currently define 2 non-static type-info
// functions, ie. the original '_aGetMyTypeInfo()' as used by the serialization code, and the '_aGetFullTypeInfo()'
// as used by the classRank code in AnyBase.cpp. We could dispense with the original version and replace calls to it
// with full-type-info calls, but that's for another day! ...

// Perfect-type-info macros for classes with no direct ancestor ...
#define _mPerfectTypeInfoHdrNA(thisClass)														\
	static _eHndl<_eInstblTypeInfo> _aMyTypeInfo();												\
	const _eFullTypeInfo *_aGetFullTypeInfo() const												\
	{																							\
		static bool init = false;																\
		static _eFullTypeInfo ti;																\
		if (!init)																				\
		{																						\
			ti = _eFullTypeInfo( thisClass:: _aMyTypeInfo(), 1, _mNullPtr(_eFullTypeInfo) );	\
			init = true;																		\
		}																						\
		return &ti;																				\
	}

// Perfect-type-info macros for a classes that do have a direct ancestor ...
#define _mPerfectTypeInfoHdrWA(thisClass, generation, parentClass)								\
	static _eHndl<_eInstblTypeInfo> _aMyTypeInfo();												\
	const _eFullTypeInfo *_aGetFullTypeInfo() const												\
	{																							\
		static bool init = false;																\
		static _eFullTypeInfo ti;																\
		if (!init)																				\
		{																						\
			ti = _eFullTypeInfo( thisClass:: _aMyTypeInfo(), generation, parentClass::_aGetFullTypeInfo() );	\
			init = true;																		\
		}																						\
		return &ti;																				\
	}

#define _mPerfectTypeInfoFwd																	\
	_eHndl<_eInstblTypeInfo> _aGetMyTypeInfo() const											\
	{																							\
		return _aMyTypeInfo();																	\
	}

// Macros defining the interface members of enumeration classes
// We have to call the ~~ function _oLocalRank instead of _oRank so that we can refer to it within macro _mEnumWrapper (see Wrappers.hpp)
#define _mEnumMembers													\
	inline _rstring _ltoString(const _eEnum x)      					\
		{	return _lString(_lnaming[static_cast<_eInt>(x)]);	}		\
	inline _eEnum _oSucc(const _eEnum x)								\
		{	return static_cast<_eEnum>(x+1);	}						\
	inline _eEnum _oPred(const _eEnum x)								\
		{	return static_cast<_eEnum>(x-1);	}						\
	inline _eInt _oPlus(const _eEnum x)									\
		{	return static_cast<_eInt>(x);	}                           \
	inline _eSeq<_eEnum> _oRange(const _eEnum x, const _eEnum y)		\
		{                                                               \
			_eSeq<_eEnum> result = _eSeq<_eEnum>();                     \
			for (_eEnum i = x; i <= y; i = _oSucc(i))                   \
				result = result._lAppend(i);                            \
			return result;                                              \
		}

#define _mLocalEnumMembers												\
	static inline _rstring _ltoString(const _eEnum x)      				\
		{	return _lString(_lnaming[static_cast<_eInt>(x)]);	}		\
	static inline _eEnum _oSucc(const _eEnum x)							\
		{	return static_cast<_eEnum>(x+1);	}						\
	static inline _eEnum _oPred(const _eEnum x)							\
		{	return static_cast<_eEnum>(x-1);	}						\
	static inline _eInt _oPlus(const _eEnum x)							\
		{	return static_cast<_eInt>(x);	}                           \
	static inline _eSeq<_eEnum> _oRange(const _eEnum x, const _eEnum y)	\
		{                                                               \
			_eSeq<_eEnum> result = _eSeq<_eEnum>();                     \
			for (_eEnum i = x; i <= y; i = _oSucc(i))                   \
				result = result._lAppend(i);                            \
			return result;                                              \
		}

// Macro defining functionality required to be generated outside the enumeration namespace in the _1.hpp file
// This global 'toString' declaration caters for toString being called on a template parameter that is instantiated with an enumeration type.
// It also defined the relative rank operator (not just for template parameters)
#define _mEnumGlobalSupport(enumName)									\
	using enumName :: _ltoString;										\
	inline _eRank :: _eEnum _oRank(const enumName :: _eEnum x, const enumName :: _eEnum y)	\
		{                                                               \
			return (y < x) ? _eRank :: above                            \
				: (x < y) ? _eRank :: below                             \
					: _eRank :: same;                                   \
		}

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
