// **************************************************************************
// * File:   BaseHandle.hpp													*
// * Target: C++ version of Perfect Developer runtime system				*
// * Author: (C) 2001 Escher Technologies Ltd.								*
// * Desc:   Definition of base class for runtime classes '_eHandle'	 	*
// *         '_eUnion' and '_eFrom'											*
// **************************************************************************

#if !defined(_h_BaseHandle)
#define _h_BaseHandle 1

//-----------------------------------------------------------
// Forward declaration for _eAny is included here so that the
// classes that inherit from _eBaseHandle can refer to _eAny.
//-----------------------------------------------------------

class _eAny;

//--------------------------------
// Class '_eBaseHandle' definition
//--------------------------------

class _eBaseHandle
{
  protected:
	_eAnyBase *Object;

	//----------------------------
	// Constructors and destructor
	//----------------------------

	// Default constructor
	_eBaseHandle() : Object(NULL)
	{
	}

	// Copy constructor
	_eBaseHandle(const _eBaseHandle& h) : Object(h.Object)
	{
		AddLink();
	}

	// Construct from an object pointer
	explicit _eBaseHandle(const _eAnyBase* ob)
	  : Object(const_cast<_eAnyBase*>(ob))
	{
		AddLink();
	}

	// Destructor
	~_eBaseHandle()
	{
		DelLink();
		Object = NULL;
	}

	//----------------------------------------
	// Concrete protected member functionality
	//----------------------------------------

	// Increment the sharing count
	void AddLink()
	{
		if (Object != NULL)
		{
			if (Object->_lNotSharable())
			{
				Object = Object->_lCopy();
			}
			Object->_lIncrementCounter();
		}
	}

	// Decrement the sharing count
	void DelLink()
	{
		if (Object != NULL)
		{
			if (Object->_lDecrementCounter())
			{
				delete Object;
			}
		}
	}

	// Get a unique copy of the object, if it is shared,
	// by making a copy of the object and reducing the
	// use-count on the original...
    void Unbind()
	{
		if (Object != NULL && !(Object->_lIsUnique()) && !(Object->_lNotSharable()))
		{
			// Create another copy of the object on the
			// heap (use-count will be 0)...
			_eAnyBase* temp = Object->_lCopy();
			// Make it permanent by setting use-count
			temp->_lIncrementCounter();
	  #if 0	// if multi-threading supported...
			// ...another thread may have decremented the count
			if (Object->_lDecrementCounter())
			{
				delete Object;
			}
	  #else	// No multi-threading.
			// Reduce use-count on original object - note
			// it cannot reach 0 because it was not 1...
			(void) Object->_lDecrementCounter();
	  #endif
			Object = temp;
		}
	}

	// Re-assign the object pointer. We must allow for the possibility that
	// the new object is the same as the old one. Either test for this
	// condition, or increment the new object pointer's reference count
	// before decrementing the old object's reference count (if we do it
	// the other way round, we risk de-allocating the object if they are
	// both the same).
	void assign(const _eAnyBase* newObject)
	{
		// Do an AddLink() on the new object
		if (newObject != NULL)
		{
			if (newObject->_lNotSharable())
			{
				newObject = newObject->_lCopy();
			}
			newObject->_lIncrementCounter();
		}
		DelLink();
		Object = const_cast<_eAnyBase*>(newObject);
	}

	// Copy assignment operator. This is used by the system-generated copy
	// assignment operator of each derived class.
	void operator=(const _eBaseHandle& x)
	{
		assign(x.Object);
	}

	void checkNotNull() const
	{
		_mCannotHappen(Object == NULL, "Attempted to access member of null object");
	}

	void checkUnionNotNull() const
	{
		_mCannotHappen(Object == NULL, "Union with void was unexpectedly void");
	}

	void checkUnionIsNull() const
	{
		_mCannotHappen(Object != NULL, "Union with void was unexpectedly non-void");
	}

  public :
	//-------------------------------------------------------
	// Public functions (called on descendents of this class)
	//-------------------------------------------------------

	// Check whether the object is null
	bool IsNull() const
    {
    	return Object == NULL;
    }

	// Assign NULL to the handle
	void MakeNull()
	{
		DelLink();
		Object = NULL;
	}

	// Generic equality operator for comparing any pair of
	// handles, froms or unions...
	_eBool operator==(const _eBaseHandle& x) const
	{
		return
					// both point to the same object or both are null
					x.Object == Object
				|| (   Object != NULL
					&& x.Object != NULL
					&& Object->_lType() == x.Object->_lType()
					// or they point to different but equal non-null objects
					&& Object->_lEqual(x.Object)
				   );
	}

	// Generic operator for swapping two handles
	// We avoid changing use counts for efficiency
	// This must only be called for swapping two handles of the same type!
	// Called only (as at 8-12-99) from global schema 'swap' (see later in this file)
	void swap(_eBaseHandle& x)
	{
		_eAnyBase* const temp = Object;
		Object = x.Object;
		x.Object = temp;
	}

	//-------------------------------------------------------------------------------------------------------------------
	// Define store and fill for handles/froms/unions/Wrapped-types. Note that to call the _astore(..)/_afill(..) on the
	// storable type pointed to by 'Object', we just call that schema directly. We can do this because there is a dummy
	// version of _astore(..) and _afill(..) defined as virtual in _eAnybase (these dummy versions don't do anything at all).
	// This solution gets around the problems I had with trying to use 'reinterpret_cast' on 'Object' - ie. trying to cast
	// it to an _eStorable* and then calling _astore(..)/_afill(..) on the result. This cast was not possible because of
	// include order problems - the full definition of _eStorable was required in order to carry out the cast, but this
	// is impossible because of circularities in the includes.
	//-------------------------------------------------------------------------------------------------------------------

	// Schema to implement persistant storage SAVing for handles, froms and unions ...
	// Note that the 'fill' schema is defined separately in all descendents.
	void _astore(_eStoreStream* stream) const;
};

//-----------------------------------------------------------
// Specialised version of schema 'swap' for handles
// This is more efficient than using the generic template
//-----------------------------------------------------------
inline void swap(_eBaseHandle& x, _eBaseHandle& y)
{
	x.swap(y);
}

//---------------------------------------------------------------------------------------------------
// Class used to generate a temporary handle to ensure that an object will not get deleted too soon
// This is used by the "_mConstructor" macro, to avoid the situation where a constructor takes a copy
// of 'self' (e.g. by calling a member function that returns 'self'), which is then deleted, causing
// 'self' to be deleted (because the use count of the current object is zero in a constructor).
//---------------------------------------------------------------------------------------------------
class _eConstructorHandle
{
	private:
		_eAnyBase* Object;
	public:
		_eConstructorHandle(_eAnyBase *arg): Object(arg)
		{
			Object->_lIncrementCounter();
		}

		~_eConstructorHandle()
		{
			Object->_lDecrementCounter();
		}
};

//---------------------------
// End of guarded header file
//---------------------------

#endif

// End.
